/* httpdconf.java -- httpd configuration
   Copyright (C) 1999 Darrick Wong
*/
import java.io.*;
import java.util.*;

public class httpdconf {
    private int m_iDebugLevel;
    private String  m_sNormalLogFile;
    private boolean m_bOverwriteNormalLog;
    private String  m_sErrorLogFile;
    private boolean m_bOverwriteErrorLog;
    private PrintStream m_psNormalLog, m_psErrorLog;
    private Vector m_vServers;
    private ThreadPool m_tpThreadPool;

    public static final String sProdName = "DWong's HTTP Server, version 4.03";

    public int getDebugLevel()              {return m_iDebugLevel;}
    public String getNormalLogFile()        {return m_sNormalLogFile;}
    public String getErrorLogFile()         {return m_sErrorLogFile;}
    public boolean canOverwriteNormalLog()  {return m_bOverwriteNormalLog;}
    public boolean canOverwriteErrorLog()   {return m_bOverwriteErrorLog;}
    public PrintStream getNormalStream()    {return m_psNormalLog;}
    public PrintStream getErrorStream()     {return m_psErrorLog;}
    public ThreadPool getThreadPool()       {return m_tpThreadPool;}

    public void setDebugLevel(int iDebugLevel)   {m_iDebugLevel = iDebugLevel;}

    public void setNormalLogFile(String s) throws IOException {
        m_sNormalLogFile = s;
        if(m_sNormalLogFile == null) {
            setNormalStream(System.out);
        } else if(m_sNormalLogFile.equals(m_sErrorLogFile)) {
            setNormalStream(getErrorStream());
        } else {
            setNormalStream(new PrintStream(new FileOutputStream(m_sNormalLogFile)));
        }
    }

    public void setErrorLogFile(String s) throws IOException {
        m_sErrorLogFile = s;

        if(m_sErrorLogFile == null) {
            setErrorStream(System.err);
        } else if(m_sErrorLogFile.equals(m_sNormalLogFile)) {
            setErrorStream(getNormalStream());
        } else {
            setErrorStream(new PrintStream(new FileOutputStream(m_sErrorLogFile)));
        }
    }

    public void setOverwriteNormalLog(boolean b) {m_bOverwriteNormalLog = b;}
    public void setOverwriteErrorLog(boolean b)  {m_bOverwriteErrorLog = b;}
    public void setNormalStream(PrintStream ps)  {m_psNormalLog = ps;}
    public void setErrorStream(PrintStream ps)   {m_psErrorLog = ps;}
    public void setThreadPool(ThreadPool t)      {m_tpThreadPool = t;}

    public httpdconf() {
        m_psNormalLog = System.out;
        m_psErrorLog = System.err;
        m_sNormalLogFile = null;
        m_bOverwriteNormalLog = true;
        m_sErrorLogFile = null;
        m_bOverwriteErrorLog = true;
        m_vServers = new Vector();
    }

    public void println(String thread, String x) {m_psNormalLog.println(thread+": "+x);}
    public void println(String x)                {m_psNormalLog.println(x);}

    public void print(String thread, String x)   {m_psNormalLog.print(thread+": "+x);}
    public void print(String x)                  {m_psNormalLog.print(x);}

    public void errln(String thread, String x)   {m_psErrorLog.println(thread+": "+x);}
    public void errln(String x)                  {m_psErrorLog.println(x);}

    public void err(String thread, String x)     {m_psErrorLog.print(thread+": "+x);}
    public void err(String x)                    {m_psErrorLog.print(x);}    

    public void dumpException(String thread, Throwable e) {
        errln(thread, e.getMessage());
        e.printStackTrace(getErrorStream());
    }

    public void addServer(Server s) {
        m_vServers.addElement(s);
//        s.start();
    }

    public void delServer(Server s) {
        if(m_vServers.removeElement(s)) {
            s.interrupt();
        }
    }
    public void delServer(int s) {
        Server x = (Server)m_vServers.elementAt(s);
        if(x == null) return;
        m_vServers.removeElementAt(s);
        x.interrupt();        
    }

    public String getServerConfig() {
        /*  SERVER STATUS
            =============

            First, we provide links to other VFS goodies...
            Then we dump VM information...

            ...and then we start on the tree.
            At root level we have httpdconf, the granddaddy of the program.
            Write a <blockquote>, and then...
                ...iterate through the servers.  For each server,
                dump status information and write a <blockquote>.  Then...
                    ...iterate through the virtual hosts.
                    Dump the vhost data, and
                write a </blockquote>, and
            write a </blockquote>
        */

        StringBuffer result = new StringBuffer();

        //write out general header stuff
        result.append("<HTML><HEAD>\r\n");
        result.append("<TITLE>"+sProdName+"</TITLE>\r\n");
        result.append("<LINK REL=STYLESHEET HREF=\"/$VFS:/style.css\">\r\n");
        result.append("<META HTTP-EQUIV=\"pragma\" CONTENT=\"no-cache\"></HEAD>\r\n");
        result.append("\r\n");
        result.append("<H1>"+sProdName+"</H1>\r\n");

        //links to other VFS stuff
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD>\r\n");
        result.append("<H2>Links:</H2>\r\n");
        result.append("<UL>\r\n");
        result.append("<LI>To access any server's status log, browse to http://<I>server</I>/$VFS:/log.txt.  For example, <A HREF=\"/$VFS:/log.txt\">browse this server's log</A>.</LI>\r\n");
        result.append("<LI>To access any server's error log, browse to http://<I>server</I>/$VFS:/err.txt.  For example, <A HREF=\"/$VFS:/err.txt\">browse this server's log</A>.</LI>\r\n");
        result.append("<LI>To deliberately send an error, browse to http://<I>server</I>/$VFS:/sigerr?err=<I>error number</I>.  For example, <A HREF=\"/$VFS:/sigerr?err=500\">send an error</A>.</LI>\r\n");
        result.append("<UL>\r\n");
        result.append("</TD></TR></TABLE><BR><BR>\r\n");

        //java vm information
        System.gc();
        Runtime r = Runtime.getRuntime();

        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD>\r\n");
        result.append("<H2>Java VM Information:</H2>\r\n");
        result.append("<UL>\r\n");
        result.append("<LI>Free memory = "+r.freeMemory()+"</LI>\r\n");
        result.append("<LI>Total memory = "+r.totalMemory()+"</LI>\r\n");
        result.append("<LI>Used memory = "+(r.totalMemory() - r.freeMemory())+"</LI>\r\n");
        result.append("<UL>\r\n");
        result.append("</TD></TR></TABLE><BR><BR>\r\n");

        //System properties
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD>\r\n");
        result.append("<H2>System properties:</H2>\r\n");
        result.append("<UL>\r\n");
        Properties props = System.getProperties();
        Enumeration enu = props.propertyNames();
        String name;
        while(enu.hasMoreElements()) {
            try{
                name = (String) enu.nextElement();
                if(name.equals("line.separator")) continue;
                result.append("<LI>"+name+" = "+props.getProperty(name)+"</LI>\r\n");
            } catch (Exception e){;}
        }
        String temp = System.getProperty("line.separator");
        String pmet = "";
        for(int i = 0; i < temp.length(); i++) {
            char x = temp.charAt(i);
            if(x == '\r') pmet += "\\r";//pmet += "0x000D";
            else if(x == '\n') pmet += "\\n";//pmet += "0x000A";
            else pmet += x;
        }
        result.append("<LI>line.separator = "+pmet+"</LI>\r\n");
        result.append("<LI>File.pathSeparator = "+File.pathSeparator+"</LI>\r\n");
        result.append("<LI>File.pathSeparatorChar = "+File.pathSeparatorChar+"</LI>\r\n");
        result.append("<LI>File.separator = "+File.separator+"</LI>\r\n");
        result.append("<LI>File.separatorChar = "+File.separatorChar+"</LI>\r\n");
        result.append("<UL>\r\n");
        result.append("</TD></TR></TABLE><BR><BR>\r\n");

        //httpdconf dump
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD>\r\n");
        result.append("<H2>Overall httpd configuration:</H2>\r\n");
        result.append("<UL>\r\n");
        result.append("<LI>Debug level = "+m_iDebugLevel+"</LI>\r\n");
        result.append("<LI>Status log file = "+m_sNormalLogFile+"</LI>\r\n");
        result.append("<LI>Can overwrite status log? "+m_bOverwriteNormalLog+"</LI>\r\n");
        result.append("<LI>Error log file = "+m_sErrorLogFile+"</LI>\r\n");
        result.append("<LI>Can overwrite error log? "+m_bOverwriteErrorLog+"</LI>\r\n");
        result.append("<UL>\r\n");
        result.append("</TD></TR></TABLE><BR><BR>\r\n");

        //now iterate through the servers.
        Enumeration e = m_vServers.elements();
        
        while(e.hasMoreElements()) {	
			Server s = (Server)e.nextElement();
			s.dumpConfig(result);
        }

        m_tpThreadPool.dumpConfig(result);
        return result.toString();
    }

    public static httpdconf generateConfigFromStrings(BufferedReader br, SuperInt line) throws Exception {
        String input, key, value;
        int e, s;
        httpdconf hc = new httpdconf();

        while( (input=br.readLine()) != null) {
            input = input.trim();
            line.increment();

            if(input.startsWith("//") || input.startsWith(";") || input.startsWith("#") || input.equals("")) continue;

            if(input.equals("}"))
                return hc;
                
            if( (s = input.indexOf(';')) == -1) throw new Exception("Parse error, line "+line+": Semicolon not found.");
            if( (e = input.indexOf('=')) == -1) throw new Exception("Parse error, line "+line+": Equals sign not found.");
            key = input.substring(0, e).toLowerCase();
            value = input.substring(e+1, s);

            if(key.equals("debuglevel")) {
                hc.setDebugLevel(Integer.parseInt(value));
            } else if(key.equals("normallogfile")) {
                hc.setNormalLogFile(value);
            } else if(key.equals("errorlogfile")) {
                hc.setErrorLogFile(value);
            } else {
                throw new Exception("Parse error, line "+line+": Unrecognized directive \""+key+"\".");
            }

//            System.out.println("httpdconf: \""+key+"\"=\""+value+"\"");
        }
        //If we get here, we haven't gotten to a close brace--an error has occured.
        throw new Exception("Parse error, line "+line+": Closing brace not found.");
    }
}
