/* ThreadPool.java -- manages a queue of idle threads.
   Copyright (C) 1999 Darrick Wong

   NOTE: Please reenable the setters as appropriate.
*/
import java.util.*;

public class ThreadPool {
    private Vector m_vAllThreads;
    private Stack m_vIdleThreads;
    private int m_iMinThreadCount;
    private int m_iMaxThreadPoolSize;
    private int m_iMaxThreadCount;
    private int m_iActiveThreadCount;
    private boolean m_bMakeTempThreads;
    private httpdconf m_hcHttpdConf;

    public Vector getAllThreads()         {return m_vAllThreads;}
    public Stack getIdleThreads()         {return m_vIdleThreads;}
    public int getMinimumThreadCount()    {return m_iMinThreadCount;}
    public int getMaximumThreadPoolSize() {return m_iMaxThreadPoolSize;}
    public int getMaximumThreadCount()    {return m_iMaxThreadCount;}
    public int getActiveThreadCount()     {return m_iActiveThreadCount;}
    public boolean canMakeTempThreads()   {return m_bMakeTempThreads;}
    public httpdconf getHttpdConf()       {return m_hcHttpdConf;}

//    public void setAllThreads(Vector v)         {m_vAllThreads = v;}
//    public void setIdleThreads(Stack v)         {m_vIdleThreads = v;}
//    public void setMinimumThreadCount(int i)    {m_iMinThreadCount = i;}
//    public void setMaximumThreadPoolSize(int i) {m_iMaxThreadPoolSize = i;}
//    public void setMaximumThreadCount(int i)    {m_iMaxThreadCount = i;}
//    public void setMakeTempThreads(boolean b)   {m_bMakeTempThreads = b;}

    private ThreadPool() {}
    
    public ThreadPool(httpdconf hc) {
        m_iMinThreadCount = 5;
        m_iMaxThreadPoolSize = 15;//15;
        m_iMaxThreadCount = 30;
        m_bMakeTempThreads = true;
        m_hcHttpdConf = hc;
        initPool();
    }

    public ThreadPool(int minthread, int maxpool, int maxthread, boolean makethread, httpdconf hc) {
        m_iMinThreadCount = minthread;
        m_iMaxThreadPoolSize = maxpool;
        m_iMaxThreadCount = maxthread;
        m_bMakeTempThreads = makethread;
        m_hcHttpdConf = hc;
        initPool();
    }

    private void initPool() {
        m_vAllThreads = new Vector();
        m_vIdleThreads = new Stack();
        RequestHandler r;

        for(int i=0; i < m_iMinThreadCount; i++) {
            r = new RequestHandler(this, m_hcHttpdConf);
            m_vAllThreads.addElement(r);
            r.setAddedToThreadQueue(true);
            m_vIdleThreads.push(r);
        }
    }

    public RequestHandler getRequestHandler() {
        RequestHandler result;

        try {
            synchronized(m_vIdleThreads) {
                result = (RequestHandler)m_vIdleThreads.pop();
            }
        } catch (EmptyStackException e) { //Branch here if the stack of idle threads is empty.
            /*  HANDLING A FLOOD OF REQUESTS
                ============================

              When somebody wants a thread:
                If the active thread count is less than the thread limit,
                    Increment the active thread count and return a new thread.
                Otherwise, return null

              When threads come back:
                If the thread's flagged as already being in the vector,
                    put it into the idle thread queue and exit.

                If the thread vector isn't full, then
                    add it to the thread vector
                    set the added flag
                    push it onto the idle stack
                else tell it to die.
            */
            if(m_iActiveThreadCount < m_iMaxThreadCount) {
                m_iMaxThreadCount++;
                return new RequestHandler(this, m_hcHttpdConf);
            } else return null;
        }
        m_iActiveThreadCount++;
        return result;
    }

    public void returnRequestHandler(RequestHandler rh) {
        m_iActiveThreadCount--;

        if(m_hcHttpdConf.getDebugLevel() > 5) {
            m_hcHttpdConf.println(Thread.currentThread().getName(),
                "ThreadPool: Thread \""+rh.getName()+"\" returned; queue size " + m_vAllThreads.size());
        }

        if(rh.isAddedToThreadQueue()) {
            m_vIdleThreads.push(rh);
        } else if(m_vAllThreads.size() < m_iMaxThreadPoolSize) {
            m_vAllThreads.addElement(rh);
            rh.setAddedToThreadQueue(true);
            m_vIdleThreads.push(rh);

        } else { //vector full, kill.
            rh.interrupt();
        }
    }

    public void dumpConfig(StringBuffer result) {
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD>\r\n");
        result.append("<H2>Thread pool:</H2>\r\n");
        result.append("<UL>\r\n");
        result.append("<LI>Minimum size of thread pool = "+m_iMinThreadCount+"</LI>\r\n");
        result.append("<LI>Maximum size of thread pool = "+m_iMaxThreadPoolSize+"</LI>\r\n");
        result.append("<LI>Maximum number of threads = "+m_iMaxThreadCount+"</LI>\r\n");
        result.append("<LI>Active threads = "+m_iActiveThreadCount+"</LI>\r\n");
        result.append("<LI>Can make temporary threads to handle large numbers of requests? "+m_bMakeTempThreads+"</LI>\r\n");
        result.append("</UL>\r\n");

        result.append("<H2>Threads in the pool:</H2>\r\n");
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD WIDTH=50>&nbsp;</TD><TD>\r\n");

		Enumeration e = m_vAllThreads.elements();
		while(e.hasMoreElements()) {
			RequestHandler rh = (RequestHandler)e.nextElement();
			rh.dumpConfig(result);
		}

        result.append("</TD></TR></TABLE>\r\n"); //inner table
        result.append("</TD></TR></TABLE><BR><BR>\r\n");

    }
}

