/* ServerConf -- holds information for a server running on a specified port.
   Copyright (C) 1999 Darrick Wong.
*/

import java.util.*;
import java.io.*;

public class ServerConf {
    private int m_iDebugLevel;
    private int m_iPort;
    private int m_iIncomingQueueSize;
    private Vector m_vHosts;
    private long m_lTotalRequests;
    private PrintStream m_psNormalLog, m_psErrorLog;
    private String m_sSysAdminEmail;
    private long m_lTimeoutMillis;

    public int getPort()                        {return m_iPort;}
    public int getIncomingQueueSize()           {return m_iIncomingQueueSize;}
    public Vector getVirtualHosts()             {return (Vector)m_vHosts.clone();}
    public long getTotalRequests()              {return m_lTotalRequests;}
    public PrintStream getNormalStream()        {return m_psNormalLog;}
    public PrintStream getErrorStream()         {return m_psErrorLog;}
    public int getDebugLevel()                  {return m_iDebugLevel;}
    public String getSysAdminEmail()            {return m_sSysAdminEmail;}
    public long getTimeoutMillis()              {return m_lTimeoutMillis;}

    public void setPort(int iPort)                              {m_iPort = iPort;}
    public void setIncomingQueueSize(int iIncomingQueueSize)    {m_iIncomingQueueSize = iIncomingQueueSize;}
//    public void setVirtualHosts(Vector v)                     {m_vHosts = v;}
    public void incTotalRequests()                              {m_lTotalRequests++;}
    public void decTotalRequests()                              {m_lTotalRequests--;}
    public void setNormalStream(PrintStream s)                  {m_psNormalLog = s;}
    public void setErrorStream(PrintStream s)                   {m_psErrorLog = s;}
    public void setDebugLevel(int iDebugLevel)                  {m_iDebugLevel = iDebugLevel;}
    public void setSysAdminEmail(String s)                      {m_sSysAdminEmail = s;}
    public void setTimeoutMillis(long l)                        {m_lTimeoutMillis = l;}

    public void println(String thread, String x) {m_psNormalLog.println(thread+": "+x);}
    public void println(String x)                {m_psNormalLog.println(x);}

    public void print(String thread, String x)   {m_psNormalLog.print(thread+": "+x);}
    public void print(String x)                  {m_psNormalLog.print(x);}

    public void errln(String thread, String x)   {m_psErrorLog.println(thread+": "+x);}
    public void errln(String x)                  {m_psErrorLog.println(x);}

    public void err(String thread, String x)     {m_psErrorLog.print(thread+": "+x);}
    public void err(String x)                    {m_psErrorLog.print(x);}    

    public ServerConf() { //default constructor, initialize defaults...
        m_iDebugLevel = 0;
        m_iPort = 80; //serve off port 80
        m_iIncomingQueueSize = 100;  //max 100 unserviced requests before connections refused
        m_vHosts = new Vector(); //no hosts means useless server.
                                 //but don't kill in case user attaches another vhost.
        m_lTotalRequests = 0;
        m_psNormalLog = System.out;
        m_psErrorLog = System.err;
        m_sSysAdminEmail = "root@localhost";
        m_lTimeoutMillis = 30000; //30s timeout
    }

    public void dumpException(String thread, Throwable e) {
        errln(thread, e.getMessage());
        e.printStackTrace(getErrorStream());
    }

    public void addVirtualHost(VHostConf hc) {
        Enumeration e = m_vHosts.elements();
        VHostConf hcc;
        while(e.hasMoreElements()) {
            hcc = (VHostConf)e.nextElement();
            if( hcc.getHostName().equals(hc.getHostName()) ) {
                throw new IllegalArgumentException("You cannot add two virtual hosts with the same name to the same server!");
            }
        }
        m_vHosts.addElement((Object)hc);
    }
    public void delVirtualHost(VHostConf hc) {m_vHosts.removeElement((Object)hc);}
    public void delVirtualHost(int index)    {m_vHosts.removeElementAt(index);}
    
    public void dumpConfig(StringBuffer result) {
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD>\r\n");
        result.append("<H2>Server running on port "+m_iPort+":</H2>\r\n");
        result.append("<UL>\r\n");
        result.append("<LI>Debug level = "+m_iDebugLevel+"</LI>\r\n");
        result.append("<LI>Idle timeout = "+m_lTimeoutMillis+"</LI>\r\n");
        result.append("<LI>Incoming queue size = "+m_iIncomingQueueSize+"</LI>\r\n");
        result.append("<LI>System administrator's email = "+m_sSysAdminEmail+"</LI>\r\n");
        result.append("<LI>Total requests = "+m_lTotalRequests+"</LI>\r\n");
        result.append("</UL>\r\n");
        
        result.append("<H2>Virtual Hosts:</H2>\r\n");
        result.append("<TABLE CELLBORDER=2 CELLSPACING=2 WIDTH=100% BORDER=0 BGCOLOR=\"#CCCCCC\"><TR><TD WIDTH=50>&nbsp;</TD><TD>\r\n");
		
		Enumeration e = m_vHosts.elements();
		while(e.hasMoreElements()) {
			VHostConf vhc = (VHostConf)e.nextElement();
			vhc.dumpConfig(result);
		}
			
        result.append("</TD></TR></TABLE>\r\n"); //inner table
        result.append("</TD></TR></TABLE><BR><BR>\r\n");
    }

    public static ServerConf generateConfigFromStrings(BufferedReader br, SuperInt line, Hashtable vHosts, httpdconf hc) throws Exception {
        String input, key, value;
        int e, s;
        ServerConf sc = new ServerConf();
        sc.setNormalStream(hc.getNormalStream());
        sc.setErrorStream(hc.getErrorStream());

        while( (input=br.readLine()) != null) {
            input = input.trim();
            line.increment();

            if(input.startsWith("//") || input.startsWith(";") || input.startsWith("#") || input.equals("")) continue;

            if(input.equals("}"))
                return sc;

            if( (s = input.indexOf(';')) == -1) throw new Exception("Parse error, line "+line+": Semicolon not found.");
            if( (e = input.indexOf('=')) == -1) throw new Exception("Parse error, line "+line+": Equals sign not found.");
            key = input.substring(0, e).toLowerCase();
            value = input.substring(e+1, s);

            if(key.equals("port")) {
                sc.setPort(Integer.parseInt(value));

            } else if(key.equals("incomingqueuesize")) {
                sc.setIncomingQueueSize(Integer.parseInt(value));
            } else if(key.equals("debuglevel")) {
                sc.setDebugLevel(Integer.parseInt(value));
            } else if(key.equals("sysadminemail")) {
                sc.setSysAdminEmail(value);
            } else if(key.equals("timeoutmillis")) {
                sc.setTimeoutMillis(Integer.parseInt(value));

            } else if(key.equals("virtualhosts")) {
                StringTokenizer st = new StringTokenizer(value, ",", false);

                while(st.hasMoreTokens()){
                    String vhost = st.nextToken().toLowerCase();
                    VHostConf vhc = (VHostConf) vHosts.get(vhost);
                    if(vhc == null) throw new Exception("Parse error, line "+line+": Virtual host \""+vhost+"\" not found.  Possible cause: It has not been created, or its configuration was not placed BEFORE that of the server.");
                    sc.addVirtualHost(vhc);
                }

            } else {
                throw new Exception("Parse error, line "+line+": Unrecognized directive \""+key+"\".");
            }

//            System.out.println("serverconf: \""+key+"\"=\""+value+"\"");
        }
        //If we get here, we haven't gotten to a close brace--an error has occured.
        throw new Exception("Parse error, line "+line+": Closing brace not found.");
    }
}