import javax.swing.*;
import javax.swing.event.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;

import jrand.*;
import jrand.ui.*;
import jrand.event.*;
import jrand.parser.*;
import jrand.error.*;

public class CFGEditor extends JFrame {
	private CFGList cl;
	private JList rulelist;
	private Container content;
	private JMenuBar menubar;
	private File cfgfile;
	private JTree ruledefn;
	private boolean isChangingListModel;
	private JMenuItem filerevertitem, listdelruleitem, listrenruleitem;

	public final static String prodName = "CFG Editor";
	
	public static void main(String args[]) throws Exception {
		CFGEditor ce;
		if(args.length > 0) {
			ce = new CFGEditor(new File(args[0]));
		} else {
			ce = new CFGEditor();
		}
	}

	//handle window events.
	private WindowAdapter windowlistener = new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
			System.exit(0);
		}
	};

	//command pathway definitions
	private AbstractAction filenew = new AbstractAction("New") {
		public void actionPerformed(ActionEvent e) {
			newFile();
		}
	};
	
	private AbstractAction fileopen = new AbstractAction("Open") {
		public void actionPerformed(ActionEvent e) {
			JFileChooser chooser = new JFileChooser();
			String moo[] = {"cfg"};
			CFGFileFilter eff = new CFGFileFilter(
				moo, "Context Free Grammars");
			chooser.addChoosableFileFilter(eff);
			int result = chooser.showOpenDialog(null);
			if(result == JFileChooser.APPROVE_OPTION) {
				loadFile(chooser.getSelectedFile());
			}
		}
	};

	private AbstractAction filesave = new AbstractAction("Save") {
		public void actionPerformed(ActionEvent e) {
			if(cfgfile != null) {
				saveFile(cfgfile);
			} else {
				JFileChooser chooser = new JFileChooser();
				String moo[] = {"cfg"};
				CFGFileFilter eff = new CFGFileFilter(
					moo, "Context Free Grammars");
				chooser.addChoosableFileFilter(eff);
				int result = chooser.showSaveDialog(null);
				if(result == JFileChooser.APPROVE_OPTION) {
					saveFile(chooser.getSelectedFile());
				}
			}
		}
	};

	private AbstractAction filesaveas = new AbstractAction("Save As...") {
		public void actionPerformed(ActionEvent e) {
			JFileChooser chooser = new JFileChooser();
			String moo[] = {"cfg"};
			CFGFileFilter eff = new CFGFileFilter(
				moo, "Context Free Grammars");
			chooser.addChoosableFileFilter(eff);
			int result = chooser.showSaveDialog(null);
			if(result == JFileChooser.APPROVE_OPTION) {
				saveFile(chooser.getSelectedFile());
			}
		}
	};

	private AbstractAction filerevert = new AbstractAction("Revert") {
		public void actionPerformed(ActionEvent e) {
			loadFile(cfgfile);
		}
	};

	private AbstractAction listaddrule = new AbstractAction("Add Rule...") {
		public void actionPerformed(ActionEvent e) {
			String newname = JOptionPane.showInputDialog(null,
				"Please enter a name for the new rule.",
				"Add Rule",
				JOptionPane.QUESTION_MESSAGE
			);
			if(newname == null) return;
			try {
				cl.addRule(newname, new CFGNode(CFGNode.UNRESOLVED, "eep"));
			} catch (InvalidRuleNameException err) {
				showError("A Problem with Adding a Rule", err);
			}
			filerevertitem.setEnabled(true);
		}
	};

	private AbstractAction listrenrule = new AbstractAction("Rename Rule...") {
		public void actionPerformed(ActionEvent e) {
			String doomedrule = (String)rulelist.getSelectedValue();
			String newname = JOptionPane.showInputDialog(null,
				"Please enter a new name for the rule.",
				"Rename Rule",
				JOptionPane.QUESTION_MESSAGE
			);
			if(newname == null) return;

			try{
				cl.renameRule(doomedrule, newname);
			} catch (InvalidRuleNameException err) {
				showError("A Problem with Rule Renaming", err);
			}
			filerevertitem.setEnabled(true);
		}
	};

	private AbstractAction listdelrule = new AbstractAction("Remove Rule...") {
		public void actionPerformed(ActionEvent e) {
			String doomedrule = (String)rulelist.getSelectedValue();
			cl.deleteRule(doomedrule);
			filerevertitem.setEnabled(true);
		}
	};

	//helper methods
	private void saveFile(File fname) {
		if(fname == null) return;

		try {
			Writer grammar = new FileWriter(fname);
			grammar.write(cl.toString());
			grammar.close();

			cfgfile = fname;
			setTitle(fname.getName()+" - "+prodName);
			filerevertitem.setEnabled(false);
		} catch (Exception e) {
			showError("A Problem with Writing Grammar File", e);
		}
	}

	private void showError(String why, Exception e) {
		JOptionPane.showMessageDialog(this, e.toString(), why, JOptionPane.ERROR_MESSAGE);
		e.printStackTrace();
	}

	private void loadFile(File fname) {
		if(fname == null) {
			newFile();
			return;
		}

		isChangingListModel = true;

		try {
			Reader grammar = new FileReader(fname);
			CFGParser cfp = new CFGParser(grammar);
			cfp.parse();
			cl = cfp.getList();	
			rulelist.setModel(new CFGListModel(cl));

			cfgfile = fname;
			setTitle(fname.getName()+" - "+prodName);
			filerevertitem.setEnabled(false);
		} catch (Exception e) {
			showError("A Problem with Reading a Grammar File", e);
		}
		isChangingListModel = false;
	}

	private void newFile() {
		isChangingListModel = true;
		cl = new CFGList();
		rulelist.setModel(new CFGListModel(cl));
		
		cfgfile = null;
		setTitle("Untitled - "+prodName);
		isChangingListModel = false;
	}

	public CFGEditor(File grammarfile) {
		super(prodName);
		initialize(grammarfile);
	}

	public CFGEditor() {
		super(prodName);
		initialize(null);
	}

	private void initialize(File grammarfile) {
		isChangingListModel = false;
		cfgfile = grammarfile;
		//initialize GUI components
		content = getContentPane();
		content.setLayout(new BorderLayout());

		rulelist = new JList();
		rulelist.addListSelectionListener(rulelistlistener);
		JPanel lpanel = new JPanel();
		lpanel.setLayout(new BorderLayout());
		lpanel.add(new JLabel("Rules"), BorderLayout.NORTH);
		JScrollPane lp = new JScrollPane(rulelist);
		lpanel.add(lp, BorderLayout.CENTER);

		ruledefn = new JTree();
		ruledefn.setFont(new Font("Dialog", 0, 14));
		JScrollPane sp = new JScrollPane(ruledefn);
		JPanel spanel = new JPanel();
		spanel.setLayout(new BorderLayout());
		spanel.add(new JLabel("Rule Definitions"), BorderLayout.NORTH);
		spanel.add(sp, BorderLayout.CENTER);

		JSplitPane jsp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, false, lpanel, spanel);
		content.add(jsp, BorderLayout.CENTER);

		//initialize command pathways
		menubar = new JMenuBar();

		JMenu filemenu = new JMenu("File", true);
		menubar.add(filemenu);

		filemenu.add(filenew);
		filemenu.add(fileopen);
		filemenu.add(filesave);
		filemenu.add(filesaveas);
		filerevertitem = filemenu.add(filerevert);
		filemenu.addSeparator();

		filemenu.add(new AbstractAction("Exit") {
			public void actionPerformed(ActionEvent e) {
				System.exit(0);
			}
		});

		JMenu listmenu = new JMenu("List", true);
		menubar.add(listmenu);

		listmenu.add(listaddrule);
		listrenruleitem = listmenu.add(listrenrule);
		listrenruleitem.setEnabled(false);
		listdelruleitem = listmenu.add(listdelrule);
		listdelruleitem.setEnabled(false);

		setJMenuBar(menubar);

		//load inital data
		loadFile(grammarfile);

		//show window
		addWindowListener(windowlistener);
		setBounds(80, 64, 960, 768);
		show();
	}

	private ListSelectionListener rulelistlistener = new ListSelectionListener() {
		public void valueChanged(ListSelectionEvent e) {
			listdelruleitem.setEnabled((rulelist.getSelectedIndex() != -1));
			listrenruleitem.setEnabled((rulelist.getSelectedIndex() != -1));

			if(isChangingListModel) return;
			//ruledefn.setText(cl.getRule((String)rulelist.getSelectedValue()).toString());
			//String key = (String)rulelist.getSelectedValue();
			//CFGNode cn = cl.getRule(key);
			//ruledefn.setModel(new CFGNodeModel(cn));
		}
	};
}
