import java.io.*;
import javax.swing.tree.*;

public class AttributeInfo implements ElementInfo {
	int count;
	ConstantPoolInfo constants;
	AttributeItem attributes[];

	public AttributeInfo(ConstantPoolInfo c, DataInputStream is)
	{
		try
		{
			constants = c;
			count = is.readUnsignedShort();
			if ( count != 0 )
			{
				attributes = new AttributeItem[count];
				for ( int x = 0; x < count; x++ )
					attributes[x] = AttributeItem.extractAttribute(c, is);
			}
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
		}
	}

	public DefaultMutableTreeNode Describe() {
		if(count == 0) return new DefaultMutableTreeNode("No attributes.");

		DefaultMutableTreeNode result = new DefaultMutableTreeNode("Attributes: "+count+" items.");

		for ( int x = 0; x < attributes.length; x++)
		{
			result.add( attributes[x].Describe() );
		}
		return result;
	}
}

class AttributeItem implements ElementInfo {
	int type_index;
	int length;
	String id;
	ConstantPoolInfo constants;

	public AttributeItem(ConstantPoolInfo c, int index, String s, DataInputStream is)
	{
		constants = c;
		type_index = index;
		id = s;
		try
		{
			length = is.readInt();
			is.skipBytes(length);
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
		}
	}

	public AttributeItem()
	{
		type_index = -1;
		length = 0;
		id = "Error";
	}

	public DefaultMutableTreeNode Describe() {
		return new DefaultMutableTreeNode(id + " ["+length+" bytes]");
	}

	public static AttributeItem extractAttribute(ConstantPoolInfo c, DataInputStream is)
	{
		AttributeItem a;
		int t;
		String s;

		try
		{
			t = is.readUnsignedShort();
			s = c.getUtf8(t);
		}
		catch ( IOException ioe )
		{
			t = -1;
			s = "Error";
		}

		if ( s.equals("ConstantValue") )
			a = new ConstantValueAttr(c, t, s, is);
		else if ( s.equals("SourceFile") )
			a = new SourceFileAttr(c, t, s, is);
		else if ( s.equals("Code") )
			a = new CodeAttr(c, t, s, is);
		else if ( s.equals("Exceptions") )
			a = new ExceptionsAttr(c, t, s, is);
		else if ( s.equals("LineNumberTable") )
			a = new LineNumberTableAttr(c, t, s, is);
		else if ( s.equals("LocalVariableTable") )
			a = new LocalVariableTableAttr(c, t, s, is);
		else if ( s.equals("Error") )
			a = new AttributeItem();
		else
			a = new AttributeItem(c, t, s, is);
		return a;
	}
}

class SourceFileAttr extends AttributeItem
{
	int file_index;

	public SourceFileAttr(ConstantPoolInfo c, int index, String s, DataInputStream is)
	{
		constants = c;
		id = s;
		try
		{
			length = is.readInt();
			file_index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
		}
	}

	public DefaultMutableTreeNode Describe() {
		return new DefaultMutableTreeNode(id + "(" + constants.getUtf8(file_index) + ")");
	}
}

class ConstantValueAttr extends AttributeItem
{
	public ConstantValueAttr(ConstantPoolInfo c, int index, String s, DataInputStream is) {
		super(c, index, s, is);
	}
}

class CodeAttr extends AttributeItem
{
	int max_stack, max_locals;
	int code_length;
	ByteCodeInfo code;
	AttributeInfo attributes;
	ExceptionInfo exceptions;

	public CodeAttr(ConstantPoolInfo c, int index, String s, DataInputStream is) {
		constants = c;
		id = s;
		try
		{
			length = is.readInt();
			max_stack = is.readUnsignedShort();
			max_locals = is.readUnsignedShort();
			code = new ByteCodeInfo(c, is);
			exceptions = new ExceptionInfo(c, is);
			attributes = new AttributeInfo(c, is);
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
		}
	}

	public DefaultMutableTreeNode Describe() {
		DefaultMutableTreeNode result = new DefaultMutableTreeNode(id);
		result.add(new DefaultMutableTreeNode("max_stack = "+max_stack));
		result.add(new DefaultMutableTreeNode("max_locals = "+max_locals));
		result.add(code.Describe());
		result.add(exceptions.Describe());
		result.add(attributes.Describe());
		return result;
	}

}

class ExceptionsAttr extends AttributeItem
{
	public ExceptionsAttr(ConstantPoolInfo c, int index, String s, DataInputStream is) {
		super(c, index, s, is);
	}
}

class LineNumberTableAttr extends AttributeItem
{
	public LineNumberTableAttr(ConstantPoolInfo c, int index, String s, DataInputStream is) {
		super(c, index, s, is);
	}
}

class LocalVariableTableAttr extends AttributeItem
{
	public LocalVariableTableAttr(ConstantPoolInfo c, int index, String s, DataInputStream is) {
		super(c, index, s, is);
	}
}

class ExceptionInfo implements ElementInfo
{
	int count;
	ExceptionItem exceptions[];

	public ExceptionInfo(ConstantPoolInfo c, DataInputStream is)
	{
		try
		{
			count = is.readUnsignedShort();
			if ( count != 0 )
			{
				exceptions = new ExceptionItem[ count ];
				for ( int xx = 0; xx < count; xx++ )
					exceptions[xx] = new ExceptionItem(c, is);
			}
		}
		catch ( IOException ioe )
		{
			System.out.println("" + ioe);
		}
	}

	public DefaultMutableTreeNode Describe() {
		if(count == 0) return new DefaultMutableTreeNode("No exceptions.");

		DefaultMutableTreeNode result = new DefaultMutableTreeNode("Exceptions: "+count+" items.");
		for ( int x = 0; x < exceptions.length; x++)
		{
			result.add(exceptions[x].Describe());
		}
		return result;
	}
}

class ExceptionItem implements ElementInfo
{
	int start_pc;
	int end_pc;
	int handler_pc;
	int catch_type;
	ConstantPoolInfo constants;

	public ExceptionItem(ConstantPoolInfo c, DataInputStream is)
	{
		constants = c;
		try
		{
			start_pc = is.readUnsignedShort();
			end_pc = is.readUnsignedShort();
			handler_pc = is.readUnsignedShort();
			catch_type = is.readUnsignedShort();
		}
		catch ( IOException ioe )
		{
			System.out.println("" + ioe);
		}
	}

	public DefaultMutableTreeNode Describe() {
		if(catch_type == 0) return new DefaultMutableTreeNode("All exceptions.");

		DefaultMutableTreeNode result = new DefaultMutableTreeNode(constants.getName(catch_type));
		result.add(new DefaultMutableTreeNode("start_pc = "+start_pc));
		result.add(new DefaultMutableTreeNode("end_pc = "+end_pc));
		result.add(new DefaultMutableTreeNode("handler_pc = "+handler_pc));
		return result;
	}

}
