/**
 * Set up the grid to plot fractals.
 * Copyright (C)2003 Darrick Wong
 */
package grid.client.fractal;

import grid.util.Defaults;
import grid.manager.Job;
import grid.manager.ClientManager;
import grid.manager.JobFinishedCallback;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.awt.image.BufferedImage;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;
import java.rmi.Naming;
import java.io.Serializable;
import java.io.File;
import javax.imageio.ImageIO;

import java.awt.*;
import java.awt.image.*;

/**
 * Submit fractal plot jobs to the grid engine.
 * The output is written as a PNG.  Hopefully.
 */
public class App {
	private String host;
	private BufferedImage bi;
	private Map jobs;
	private int pixels[][];
	//private int pixels[];

	private static final int CHUNK_X = 256;
	private static final int CHUNK_Y = 256;

	private static final int IMAGE_X = 2560;
	private static final int IMAGE_Y = 2048;

	public static void main(String args[]) {
		String host = ".";
		if(args.length > 0) {
			host = args[0];
		}
		
		System.setSecurityManager(new java.rmi.RMISecurityManager());
		App a = new App(host);
		a.begin();
	}

	public App(String host) {
		this.host = host;
		this.jobs = new HashMap();
		bi = new BufferedImage(IMAGE_X, IMAGE_Y, BufferedImage.TYPE_INT_ARGB);
		pixels = new int[IMAGE_Y][IMAGE_X];
		//pixels = new int[IMAGE_Y * IMAGE_X];
	}

	public void begin() {
		String name = "//"+host+"/"+Defaults.managerService;

		ClientManager cm = null;
		try {
			cm = (ClientManager) Naming.lookup(name);
		} catch (Exception e) {
			System.err.println("Unable to find manager at "+host
				+"!	Aborting.");
			System.err.println(e.getMessage());
			return;
		}
		
		// Create jobs
		long createStart = System.currentTimeMillis();
		ReturnHandler rh = null;
		try {
			rh = new ReturnHandler();
		} catch (Exception e) {
			System.err.println("Couldn't create function return "
				+ "handler!  Aborting.");
			System.err.println(e.getMessage());
			return;
		}

		List jobList = new LinkedList();
		for(int j = 0; j < IMAGE_Y; j += CHUNK_Y) {
			for(int i = 0; i < IMAGE_X; i += CHUNK_X) {
				Job jb = new Job(
					new MandelPlot(
						i,
						j,
						IMAGE_X,
						IMAGE_Y,
						CHUNK_X,
						CHUNK_Y,
						-0.757, -0.631,
						-0.350, -0.080,
						/*
						-2.0, 1.0,
						-1.25, 1.25,
						*/
						-0.75
					),
					"mandel-"+i+"-"+j,
					rh,
					60000
				);

				jobList.add(jb);
				jobs.put(jb.jobID, jb);
			}
		}

		long dispatchStart = System.currentTimeMillis();
		// Now send the jobs off
		try {
			cm.scheduleJobs((Job[]) jobList.toArray(new Job[]{}));
		} catch (Exception e) {
			System.err.println("Unable to submit jobs to grid manager!");
			System.err.println(e.getMessage());
		}
		
		jobList = null;

		// Wait until everything is done...
		while(jobs.isEmpty() == false) {
			try {
				synchronized (jobs) {
					jobs.wait();
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

		long convertStart = System.currentTimeMillis();
		// Convert the pixel buffer into an image
		
		for(int iy = 0; iy < IMAGE_Y; iy++) {
			for(int ix = 0; ix < IMAGE_X; ix++) {
				bi.setRGB(ix, iy, pixels[iy][ix]);
			}
		}
		
		/*
		MemoryImageSource mis = new MemoryImageSource(IMAGE_X, IMAGE_Y, pixels, 0, 4);
		Frame f = new Frame();
		Image i = f.createImage(mis);
		Graphics2D g = bi.createGraphics();
		g.drawImage(i, 0, 0, null);
		*/

		long saveStart = System.currentTimeMillis();
		//Now take the image and save it.
		System.out.println("Finished; writing output...");
		try {
			ImageIO.write(bi, "png", new File("output.png"));
		} catch (Exception e) {
			System.err.println("Error saving PNG!");
			System.err.println(e.getMessage());
		}
		long now = System.currentTimeMillis();

		System.out.print(
			"Job Creation: " + (dispatchStart - createStart) + "ms\n"
			+ "Job Run: " + (convertStart - dispatchStart) + "ms\n"
			+ "Conversion: " + (saveStart - convertStart) + "ms\n"
			+ "Save: " + (now - saveStart) + "ms\n");
		
		System.exit(0);
	}

	class ReturnHandler extends UnicastRemoteObject implements
		Serializable, JobFinishedCallback
	{
		public ReturnHandler() throws RemoteException {
			super();
		}

		public void jobFinished(String jobID, Object result) {
			//Pull this job
			Job j = null;
			synchronized (jobs) {
				j = (Job) jobs.get(jobID);

				if(j == null) return;

				jobs.remove(jobID);
			}

			//extract job info
			MandelPlot mp = (MandelPlot) j.work;
			int x = mp.getX();
			int y = mp.getY();
			int w = mp.getWidth();
			int h = mp.getHeight();

			int pels[][] = (int[][]) result;

			
			synchronized (pixels) {
				for(int iy = 0; iy < h; iy++) {
					System.arraycopy(pels[iy], 0, pixels[iy + y], x, w);
					//System.arraycopy(pels[iy], 0, pixels, (iy + y) * IMAGE_X + x, w);
				}
			}

			synchronized (jobs) {
				jobs.notify();
			}
		}
	}
}
