/* blockdiff -- print out block numbers that differ (and aren't zeroed) */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#define NUM_BLOCKS 8192

static inline size_t min(size_t a, size_t b)
{
	return (a < b ? a : b);
}

int blocks_differ(unsigned char *a, unsigned char *b, size_t block_num,
		  size_t block_size)
{
	unsigned long *la = &a[block_num * block_size];
	unsigned long *lb = &b[block_num * block_size];
	size_t i, end;

	end = block_size / sizeof(unsigned long);
	/* First check to see if b is all zeroes */
	for (i = 0; i < end; i++)
		if (lb[i])
			goto not_zero;

	for (i = end * sizeof(unsigned long); i < block_size; i++)
		if (b[i])
			goto not_zero;
	return 0;

not_zero:
	/* Now compare a and b */
	for (i = 0; i < end; i++)
		if (la[i] != lb[i])
			return 1;

	for (i = end * sizeof(unsigned long); i < block_size; i++)
		if (a[i] != b[i])
			return 1;

	return 0;
}

int compare_files(size_t block_size, const char *a, const char *b)
{
	FILE *fpa, *fpb;
	unsigned char *ma, *mb;
	int res = 0;
	size_t ra, rb, i, compare_limit, start;

	/* open files */
	fpa = fopen(a, "r");
	if (!fpa) {
		perror(a);
		res = 1;
		goto out;
	}

	fpb = fopen(b, "r");
	if (!fpb) {
		perror(b);
		res = 2;
		goto out2;
	}

	/* allocate buffer */
	ma = malloc(block_size * NUM_BLOCKS);
	if (!ma) {
		perror("ma");
		goto out3;
	}

	mb = malloc(block_size * NUM_BLOCKS);
	if (!mb) {
		perror("mb");
		goto out4;
	}

	/* Run comparison */
	start = 0;
	while (1) {
		ra = fread(ma, block_size, NUM_BLOCKS, fpa);
		rb = fread(mb, block_size, NUM_BLOCKS, fpb);

		if (ra != rb)
			fprintf(stderr, "File size mismatch!\n");

		compare_limit = min(ra, rb);
		for (i = 0; i < compare_limit; i++)
			if (blocks_differ(ma, mb, i, block_size))
				printf("%lu\n", start + i);

		if (ra < NUM_BLOCKS || rb < NUM_BLOCKS)
			break;

		start += ra;
	}

	/* tear down */
	free(mb);
out4:
	free(ma);
out3:
	fclose(fpb);
out2:
	fclose(fpa);
out:

	return res;
}

int main(int argc, char *argv[])
{
	size_t block_size;

	if (argc != 4) {
		printf("Usage: %s blocksize file_a file_b\n", argv[0]);
		return 0;
	}

	block_size = strtoul(argv[1], NULL, 0);
	if (errno) {
		perror(argv[1]);
		return 1;
	}

	return compare_files(block_size, argv[2], argv[3]);
}
